/*
 * Copyright (C) 2010 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#ifndef __ASM_ARCH_MXC_MX53_PINS_H__
#define __ASM_ARCH_MXC_MX53_PINS_H__

/*!
 * @file arch-mxc/mx53_pins.h
 *
 * @brief MX53 I/O Pin List
 *
 * @ingroup GPIO_MX53
 */

#ifndef __ASSEMBLY__

/*!
 * @name IOMUX/PAD Bit field definitions
 */

/*! @{ */

/*!
 * In order to identify pins more effectively, each mux-controlled pin's
 * enumerated value is constructed in the following way:
 *
 * -------------------------------------------------------------------
 * 31-29 | 28 - 24 |  23 - 21 | 20  - 10| 9 - 0
 * -------------------------------------------------------------------
 * IO_P  |  IO_I  | GPIO_I | PAD_I  | MUX_I
 * -------------------------------------------------------------------
 *
 * Bit 0 to 9 contains MUX_I used to identify the register
 * offset (0-based. base is IOMUX_module_base) defined in the Section
 * "sw_pad_ctl & sw_mux_ctl details" of the IC Spec. The
 * similar field definitions are used for the pad control register.
 * For example, the MX53_PIN_GPIO_19 is defined in the enumeration:
 *    ( (0x20 - MUX_I_START) << MUX_I)|( (0x348 - PAD_I_START) << PAD_I)
 * It means the mux control register is at register offset 0x20. The pad control
 * register offset is: 0x348 and also occupy the least significant bits
 * within the register.
 */

/*!
 * Starting bit position within each entry of \b iomux_pins to represent the
 * MUX control register offset
 */
#define MUX_I			0
/*!
 * Starting bit position within each entry of \b iomux_pins to represent the
 * PAD control register offset
 */
#define PAD_I			10
/*!
 * Starting bit position within each entry of \b iomux_pins to represent which
 * mux mode is for GPIO (0-based)
 */
#define GPIO_I			21

#define MUX_IO_P                29
#define MUX_IO_I                24

#define NON_GPIO_PORT		0x7
#define PIN_TO_MUX_MASK		((1 << (PAD_I - MUX_I)) - 1)
#define PIN_TO_PAD_MASK		((1 << (GPIO_I - PAD_I)) - 1)
#define PIN_TO_ALT_GPIO_MASK		((1 << (MUX_IO_I - GPIO_I)) - 1)

#define NON_MUX_I		PIN_TO_MUX_MASK
#define NON_PAD_I		PIN_TO_PAD_MASK
#define MUX_I_START		0x0020
#define PAD_I_START		0x348
#define INPUT_CTL_START		0x730
#define MUX_I_END		(PAD_I_START - 4)

#define _MXC_BUILD_PIN(gp, gi, ga, mi, pi) \
	(((gp) << MUX_IO_P) | ((gi) << MUX_IO_I) | \
	((mi) << MUX_I) | \
	((pi - PAD_I_START) << PAD_I) | \
	((ga) << GPIO_I))

#define _MXC_BUILD_GPIO_PIN(gp, gi, ga, mi, pi) \
    _MXC_BUILD_PIN(gp, gi, ga, mi, pi)

#define _MXC_BUILD_NON_GPIO_PIN(mi, pi) \
    _MXC_BUILD_PIN(NON_GPIO_PORT, 0, 0, mi, pi)

#define PIN_TO_IOMUX_MUX(pin)	((pin >> MUX_I) & PIN_TO_MUX_MASK)
#define PIN_TO_IOMUX_PAD(pin)	((pin >> PAD_I) & PIN_TO_PAD_MASK)
#define PIN_TO_ALT_GPIO(pin)	((pin >> GPIO_I) & PIN_TO_ALT_GPIO_MASK)
#define PIN_TO_IOMUX_INDEX(pin)	(PIN_TO_IOMUX_MUX(pin) >> 2)

/*! @} End IOMUX/PAD Bit field definitions */

/*!
 * This enumeration is constructed based on the Section
 * "sw_pad_ctl & sw_mux_ctl details" of the MX53 IC Spec. Each enumerated
 * value is constructed based on the rules described above.
 */
enum iomux_pins {
	MX53_PIN_GPIO_19  = _MXC_BUILD_GPIO_PIN(3, 5, 1, 0x20, 0x348),
	MX53_PIN_KEY_COL0 = _MXC_BUILD_GPIO_PIN(3, 6, 1, 0x24, 0x34C),
	MX53_PIN_KEY_ROW0 = _MXC_BUILD_GPIO_PIN(3, 7, 1, 0x28, 0x350),
	MX53_PIN_KEY_COL1 = _MXC_BUILD_GPIO_PIN(3, 8, 1, 0x2C, 0x354),
	MX53_PIN_KEY_ROW1 = _MXC_BUILD_GPIO_PIN(3, 9, 1, 0x30, 0x358),
	MX53_PIN_KEY_COL2 = _MXC_BUILD_GPIO_PIN(3, 10, 1, 0x34, 0x35C),
	MX53_PIN_KEY_ROW2 = _MXC_BUILD_GPIO_PIN(3, 11, 1, 0x38, 0x360),
	MX53_PIN_KEY_COL3 = _MXC_BUILD_GPIO_PIN(3, 12, 1, 0x3C, 0x364),
	MX53_PIN_KEY_ROW3 = _MXC_BUILD_GPIO_PIN(3, 13, 1, 0x40, 0x368),
	MX53_PIN_KEY_COL4 = _MXC_BUILD_GPIO_PIN(3, 14, 1, 0x44, 0x36C),
	MX53_PIN_KEY_ROW4 = _MXC_BUILD_GPIO_PIN(3, 15, 1, 0x48, 0x370),
	MX53_PIN_NVCC_KEYPAD = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x374),
	MX53_PIN_DI0_DISP_CLK = _MXC_BUILD_GPIO_PIN(3, 16, 1, 0x4C, 0x378),
	MX53_PIN_DI0_PIN15 = _MXC_BUILD_GPIO_PIN(3, 17, 1, 0x50, 0x37C),
	MX53_PIN_DI0_PIN2 = _MXC_BUILD_GPIO_PIN(3, 18, 1, 0x54, 0x380),
	MX53_PIN_DI0_PIN3 = _MXC_BUILD_GPIO_PIN(3, 19, 1, 0x58, 0x384),
	MX53_PIN_DI0_PIN4 = _MXC_BUILD_GPIO_PIN(3, 20, 1, 0x5C, 0x388),
	MX53_PIN_DISP0_DAT0 = _MXC_BUILD_GPIO_PIN(3, 21, 1, 0x60, 0x38C),
	MX53_PIN_DISP0_DAT1 = _MXC_BUILD_GPIO_PIN(3, 22, 1, 0x64, 0x390),
	MX53_PIN_DISP0_DAT2 = _MXC_BUILD_GPIO_PIN(3, 23, 1, 0x68, 0x394),
	MX53_PIN_DISP0_DAT3 = _MXC_BUILD_GPIO_PIN(3, 24, 1, 0x6C, 0x398),
	MX53_PIN_DISP0_DAT4 = _MXC_BUILD_GPIO_PIN(3, 25, 1, 0x70, 0x39C),
	MX53_PIN_DISP0_DAT5 = _MXC_BUILD_GPIO_PIN(3, 26, 1, 0x74, 0x3A0),
	MX53_PIN_DISP0_DAT6 = _MXC_BUILD_GPIO_PIN(3, 27, 1, 0x78, 0x3A4),
	MX53_PIN_DISP0_DAT7 = _MXC_BUILD_GPIO_PIN(3, 28, 1, 0x7C, 0x3A8),
	MX53_PIN_DISP0_DAT8 = _MXC_BUILD_GPIO_PIN(3, 29, 1, 0x80, 0x3AC),
	MX53_PIN_DISP0_DAT9 = _MXC_BUILD_GPIO_PIN(3, 30, 1, 0x84, 0x3B0),
	MX53_PIN_DISP0_DAT10 = _MXC_BUILD_GPIO_PIN(3, 31, 1, 0x88, 0x3B4),
	MX53_PIN_DISP0_DAT11 = _MXC_BUILD_GPIO_PIN(4, 5, 1, 0x8C, 0x3B8),
	MX53_PIN_DISP0_DAT12 = _MXC_BUILD_GPIO_PIN(4, 6, 1, 0x90, 0x3BC),
	MX53_PIN_DISP0_DAT13 = _MXC_BUILD_GPIO_PIN(4, 7, 1, 0x94, 0x3C0),
	MX53_PIN_DISP0_DAT14 = _MXC_BUILD_GPIO_PIN(4, 8, 1, 0x98, 0x3C4),
	MX53_PIN_DISP0_DAT15 = _MXC_BUILD_GPIO_PIN(4, 9, 1, 0x9C, 0x3C8),
	MX53_PIN_DISP0_DAT16 = _MXC_BUILD_GPIO_PIN(4, 10, 1, 0xA0, 0x3CC),
	MX53_PIN_DISP0_DAT17 = _MXC_BUILD_GPIO_PIN(4, 11, 1, 0xA4, 0x3D0),
	MX53_PIN_DISP0_DAT18 = _MXC_BUILD_GPIO_PIN(4, 12, 1, 0xA8, 0x3D4),
	MX53_PIN_DISP0_DAT19 = _MXC_BUILD_GPIO_PIN(4, 13, 1, 0xAC, 0x3D8),
	MX53_PIN_DISP0_DAT20 = _MXC_BUILD_GPIO_PIN(4, 14, 1, 0xB0, 0x3DC),
	MX53_PIN_DISP0_DAT21 = _MXC_BUILD_GPIO_PIN(4, 15, 1, 0xB4, 0x3E0),
	MX53_PIN_DISP0_DAT22 = _MXC_BUILD_GPIO_PIN(4, 16, 1, 0xB8, 0x3E4),
	MX53_PIN_DISP0_DAT23 = _MXC_BUILD_GPIO_PIN(4, 17, 1, 0xBC, 0x3E8),
	MX53_PIN_CSI0_PIXCLK = _MXC_BUILD_GPIO_PIN(4, 18, 1, 0xC0, 0x3EC),
	MX53_PIN_CSI0_MCLK = _MXC_BUILD_GPIO_PIN(4, 19, 1, 0xC4, 0x3F0),
	MX53_PIN_CSI0_DATA_EN = _MXC_BUILD_GPIO_PIN(4, 20, 1, 0xC8, 0x3F4),
	MX53_PIN_CSI0_VSYNC = _MXC_BUILD_GPIO_PIN(4, 21, 1, 0xCC, 0x3F8),
	MX53_PIN_CSI0_D4 = _MXC_BUILD_GPIO_PIN(4, 22, 1, 0xD0, 0x3FC),
	MX53_PIN_CSI0_D5 = _MXC_BUILD_GPIO_PIN(4, 23, 1, 0xD4, 0x400),
	MX53_PIN_CSI0_D6 = _MXC_BUILD_GPIO_PIN(4, 24, 1, 0xD8, 0x404),
	MX53_PIN_CSI0_D7 = _MXC_BUILD_GPIO_PIN(4, 25, 1, 0xDC, 0x408),
	MX53_PIN_CSI0_D8 = _MXC_BUILD_GPIO_PIN(4, 26, 1, 0xE0, 0x40C),
	MX53_PIN_CSI0_D9 = _MXC_BUILD_GPIO_PIN(4, 27, 1, 0xE4, 0x410),
	MX53_PIN_CSI0_D10 = _MXC_BUILD_GPIO_PIN(4, 28, 1, 0xE8, 0x414),
	MX53_PIN_CSI0_D11 = _MXC_BUILD_GPIO_PIN(4, 29, 1, 0xEC, 0x418),
	MX53_PIN_CSI0_D12 = _MXC_BUILD_GPIO_PIN(4, 30, 1, 0xF0, 0x41C),
	MX53_PIN_CSI0_D13 = _MXC_BUILD_GPIO_PIN(4, 31, 1, 0xF4, 0x420),
	MX53_PIN_CSI0_D14 = _MXC_BUILD_GPIO_PIN(5, 0, 1, 0xF8, 0x424),
	MX53_PIN_CSI0_D15 = _MXC_BUILD_GPIO_PIN(5, 1, 1, 0xFC, 0x428),
	MX53_PIN_CSI0_D16 = _MXC_BUILD_GPIO_PIN(5, 2, 1, 0x100, 0x42C),
	MX53_PIN_CSI0_D17 = _MXC_BUILD_GPIO_PIN(5, 3, 1, 0x104, 0x430),
	MX53_PIN_CSI0_D18 = _MXC_BUILD_GPIO_PIN(5, 4, 1, 0x108, 0x434),
	MX53_PIN_CSI0_D19 = _MXC_BUILD_GPIO_PIN(5, 5, 1, 0x10C, 0x438),
	MX53_PIN_NVCC_CSI0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x43C),
	MX53_PIN_JTAG_TMS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x440),
	MX53_PIN_JTAG_MOD = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x444),
	MX53_PIN_JTAG_TRSTB = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x448),
	MX53_PIN_JTAG_TDI = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x44C),
	MX53_PIN_JTAG_TCK = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x450),
	MX53_PIN_JTAG_TDO = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x454),
	MX53_PIN_EIM_A25 = _MXC_BUILD_GPIO_PIN(4, 2, 1, 0x110, 0x458),
	MX53_PIN_EIM_EB2 = _MXC_BUILD_GPIO_PIN(1, 30, 1, 0x114, 0x45C),
	MX53_PIN_EIM_D16 = _MXC_BUILD_GPIO_PIN(2, 16, 1, 0x118, 0x460),
	MX53_PIN_EIM_D17 = _MXC_BUILD_GPIO_PIN(2, 17, 1, 0x11C, 0x464),
	MX53_PIN_EIM_D18 = _MXC_BUILD_GPIO_PIN(2, 18, 1, 0x120, 0x468),
	MX53_PIN_EIM_D19 = _MXC_BUILD_GPIO_PIN(2, 19, 1, 0x124, 0x46C),
	MX53_PIN_EIM_D20 = _MXC_BUILD_GPIO_PIN(2, 20, 1, 0x128, 0x470),
	MX53_PIN_EIM_D21 = _MXC_BUILD_GPIO_PIN(2, 21, 1, 0x12C, 0x474),
	MX53_PIN_EIM_D22 = _MXC_BUILD_GPIO_PIN(2, 22, 1, 0x130, 0x478),
	MX53_PIN_EIM_D23 = _MXC_BUILD_GPIO_PIN(2, 23, 1, 0x134, 0x47C),
	MX53_PIN_EIM_EB3 = _MXC_BUILD_GPIO_PIN(1, 31, 1, 0x138, 0x480),
	MX53_PIN_EIM_D24 = _MXC_BUILD_GPIO_PIN(2, 24, 1, 0x13C, 0x484),
	MX53_PIN_EIM_D25 = _MXC_BUILD_GPIO_PIN(2, 25, 1, 0x140, 0x488),
	MX53_PIN_EIM_D26 = _MXC_BUILD_GPIO_PIN(2, 26, 1, 0x144, 0x48C),
	MX53_PIN_EIM_D27 = _MXC_BUILD_GPIO_PIN(2, 27, 1, 0x148, 0x490),
	MX53_PIN_EIM_D28 = _MXC_BUILD_GPIO_PIN(2, 28, 1, 0x14C, 0x494),
	MX53_PIN_EIM_D29 = _MXC_BUILD_GPIO_PIN(2, 29, 1, 0x150, 0x498),
	MX53_PIN_EIM_D30 = _MXC_BUILD_GPIO_PIN(2, 30, 1, 0x154, 0x49C),
	MX53_PIN_EIM_D31 = _MXC_BUILD_GPIO_PIN(2, 31, 1, 0x158, 0x4A0),
	MX53_PIN_NVCC_EIM1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x4A4),
	MX53_PIN_EIM_A24 = _MXC_BUILD_GPIO_PIN(4, 4, 1, 0x15C, 0x4A8),
	MX53_PIN_EIM_A23 = _MXC_BUILD_GPIO_PIN(5, 6, 1, 0x160, 0x4AC),
	MX53_PIN_EIM_A22 = _MXC_BUILD_GPIO_PIN(1, 16, 1, 0x164, 0x4B0),
	MX53_PIN_EIM_A21 = _MXC_BUILD_GPIO_PIN(1, 17, 1, 0x168, 0x4B4),
	MX53_PIN_EIM_A20 = _MXC_BUILD_GPIO_PIN(1, 18, 1, 0x16C, 0x4B8),
	MX53_PIN_EIM_A19 = _MXC_BUILD_GPIO_PIN(1, 19, 1, 0x170, 0x4BC),
	MX53_PIN_EIM_A18 = _MXC_BUILD_GPIO_PIN(1, 20, 1, 0x174, 0x4C0),
	MX53_PIN_EIM_A17 = _MXC_BUILD_GPIO_PIN(1, 21, 1, 0x178, 0x4C4),
	MX53_PIN_EIM_A16 = _MXC_BUILD_GPIO_PIN(1, 22, 1, 0x17C, 0x4C8),
	MX53_PIN_EIM_CS0 = _MXC_BUILD_GPIO_PIN(1, 23, 1, 0x180, 0x4CC),
	MX53_PIN_EIM_CS1 = _MXC_BUILD_GPIO_PIN(1, 24, 1, 0x184, 0x4D0),
	MX53_PIN_EIM_OE = _MXC_BUILD_GPIO_PIN(1, 25, 1, 0x188, 0x4D4),
	MX53_PIN_EIM_RW = _MXC_BUILD_GPIO_PIN(1, 26, 1, 0x18C, 0x4D8),
	MX53_PIN_EIM_LBA = _MXC_BUILD_GPIO_PIN(1, 27, 1, 0x190, 0x4DC),
	MX53_PIN_NVCC_EIM4 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x4E0),
	MX53_PIN_EIM_EB0 = _MXC_BUILD_GPIO_PIN(1, 28, 1, 0x194, 0x4E4),
	MX53_PIN_EIM_EB1 = _MXC_BUILD_GPIO_PIN(1, 29, 1, 0x198, 0x4E8),
	MX53_PIN_EIM_DA0 = _MXC_BUILD_GPIO_PIN(2, 0, 1, 0x19C, 0x4EC),
	MX53_PIN_EIM_DA1 = _MXC_BUILD_GPIO_PIN(2, 1, 1, 0x1A0, 0x4F0),
	MX53_PIN_EIM_DA2 = _MXC_BUILD_GPIO_PIN(2, 2, 1, 0x1A4, 0x4F4),
	MX53_PIN_EIM_DA3 = _MXC_BUILD_GPIO_PIN(2, 3, 1, 0x1A8, 0x4F8),
	MX53_PIN_EIM_DA4 = _MXC_BUILD_GPIO_PIN(2, 4, 1, 0x1AC, 0x4FC),
	MX53_PIN_EIM_DA5 = _MXC_BUILD_GPIO_PIN(2, 5, 1, 0x1B0, 0x500),
	MX53_PIN_EIM_DA6 = _MXC_BUILD_GPIO_PIN(2, 6, 1, 0x1B4, 0x504),
	MX53_PIN_EIM_DA7 = _MXC_BUILD_GPIO_PIN(2, 7, 1, 0x1B8, 0x508),
	MX53_PIN_EIM_DA8 = _MXC_BUILD_GPIO_PIN(2, 8, 1, 0x1BC, 0x50C),
	MX53_PIN_EIM_DA9 = _MXC_BUILD_GPIO_PIN(2, 9, 1, 0x1C0, 0x510),
	MX53_PIN_EIM_DA10 = _MXC_BUILD_GPIO_PIN(2, 10, 1, 0x1C4, 0x514),
	MX53_PIN_EIM_DA11 = _MXC_BUILD_GPIO_PIN(2, 11, 1, 0x1C8, 0x518),
	MX53_PIN_EIM_DA12 = _MXC_BUILD_GPIO_PIN(2, 12, 1, 0x1CC, 0x51C),
	MX53_PIN_EIM_DA13 = _MXC_BUILD_GPIO_PIN(2, 13, 1, 0x1D0, 0x520),
	MX53_PIN_EIM_DA14 = _MXC_BUILD_GPIO_PIN(2, 14, 1, 0x1D4, 0x524),
	MX53_PIN_EIM_DA15 = _MXC_BUILD_GPIO_PIN(2, 15, 1, 0x1D8, 0x528),
	MX53_PIN_NANDF_WE_B = _MXC_BUILD_GPIO_PIN(5, 12, 1, 0x1DC, 0x52C),
	MX53_PIN_NANDF_RE_B = _MXC_BUILD_GPIO_PIN(5, 13, 1, 0x1E0, 0x530),
	MX53_PIN_EIM_WAIT = _MXC_BUILD_GPIO_PIN(4, 0, 1, 0x1E4, 0x534),
	MX53_PIN_EIM_BCLK = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x538),
	MX53_PIN_NVCC_EIM7 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x53C),
	MX53_PIN_LVDS1_TX3_P = _MXC_BUILD_GPIO_PIN(5, 22, 0, 0x1EC, NON_PAD_I),
	MX53_PIN_LVDS1_TX2_P = _MXC_BUILD_GPIO_PIN(5, 24, 0, 0x1F0, NON_PAD_I),
	MX53_PIN_LVDS1_CLK_P = _MXC_BUILD_GPIO_PIN(5, 26, 0, 0x1F4, NON_PAD_I),
	MX53_PIN_LVDS1_TX1_P = _MXC_BUILD_GPIO_PIN(5, 28, 0, 0x1F8, NON_PAD_I),
	MX53_PIN_LVDS1_TX0_P = _MXC_BUILD_GPIO_PIN(5, 30, 0, 0x1FC, NON_PAD_I),
	MX53_PIN_LVDS0_TX3_P = _MXC_BUILD_GPIO_PIN(6, 22, 0, 0x200, NON_PAD_I),
	MX53_PIN_LVDS0_CLK_P = _MXC_BUILD_GPIO_PIN(6, 24, 0, 0x204, NON_PAD_I),
	MX53_PIN_LVDS0_TX2_P = _MXC_BUILD_GPIO_PIN(6, 26, 0, 0x208, NON_PAD_I),
	MX53_PIN_LVDS0_TX1_P = _MXC_BUILD_GPIO_PIN(6, 28, 0, 0x20C, NON_PAD_I),
	MX53_PIN_LVDS0_TX0_P = _MXC_BUILD_GPIO_PIN(6, 30, 0, 0x210, NON_PAD_I),
	MX53_PIN_GPIO_10 = _MXC_BUILD_GPIO_PIN(3, 0, 0, 0x214, 0x540),
	MX53_PIN_GPIO_11 = _MXC_BUILD_GPIO_PIN(3, 1, 0, 0x218, 0x544),
	MX53_PIN_GPIO_12 = _MXC_BUILD_GPIO_PIN(3, 2, 0, 0x21C, 0x548),
	MX53_PIN_GPIO_13 = _MXC_BUILD_GPIO_PIN(3, 3, 0, 0x220, 0x54C),
	MX53_PIN_GPIO_14 = _MXC_BUILD_GPIO_PIN(3, 4, 0, 0x224, 0x550),
	MX53_PIN_DRAM_DQM3 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x554),
	MX53_PIN_DRAM_SDQS3 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x558),
	MX53_PIN_DRAM_SDCKE1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x55C),
	MX53_PIN_DRAM_DQM2 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x560),
	MX53_PIN_DRAM_SDODT1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x564),
	MX53_PIN_DRAM_SDQS2 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x568),
	MX53_PIN_DRAM_RESET = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x56C),
	MX53_PIN_DRAM_SDCLK1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x570),
	MX53_PIN_DRAM_CAS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x574),
	MX53_PIN_DRAM_SDCLK0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x578),
	MX53_PIN_DRAM_SDQS0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x57C),
	MX53_PIN_DRAM_SDODT0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x580),
	MX53_PIN_DRAM_DQM0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x584),
	MX53_PIN_DRAM_RAS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x588),
	MX53_PIN_DRAM_SDCKE0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x58C),
	MX53_PIN_DRAM_SDQS1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x590),
	MX53_PIN_DRAM_DQM1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x594),
	MX53_PIN_PMIC_ON_REQ = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x598),
	MX53_PIN_PMIC_STBY_REQ = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x59C),
	MX53_PIN_NANDF_CLE = _MXC_BUILD_GPIO_PIN(5, 7, 1, 0x228, 0x5A0),
	MX53_PIN_NANDF_ALE = _MXC_BUILD_GPIO_PIN(5, 8 , 1, 0x22C, 0x5A4),
	MX53_PIN_NANDF_WP_B = _MXC_BUILD_GPIO_PIN(5, 9, 1, 0x230, 0x5A8),
	MX53_PIN_NANDF_RB0 = _MXC_BUILD_GPIO_PIN(5, 10, 1, 0x234, 0x5AC),
	MX53_PIN_NANDF_CS0 = _MXC_BUILD_GPIO_PIN(5, 11, 1, 0x238, 0x5B0),
	MX53_PIN_NANDF_CS1 = _MXC_BUILD_GPIO_PIN(5, 14, 1, 0x23C, 0x5B4),
	MX53_PIN_NANDF_CS2 = _MXC_BUILD_GPIO_PIN(5, 15, 1, 0x240, 0x5B8),
	MX53_PIN_NANDF_CS3 = _MXC_BUILD_GPIO_PIN(5, 16, 1, 0x244, 0x5BC),
	MX53_PIN_NVCC_NANDF = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x5C0),
	MX53_PIN_FEC_MDIO = _MXC_BUILD_GPIO_PIN(0, 22, 1, 0x248, 0x5C4),
	MX53_PIN_FEC_REF_CLK = _MXC_BUILD_GPIO_PIN(0, 23, 1, 0x24C, 0x5C8),
	MX53_PIN_FEC_RX_ER = _MXC_BUILD_GPIO_PIN(0, 24, 1, 0x250, 0x5CC),
	MX53_PIN_FEC_CRS_DV = _MXC_BUILD_GPIO_PIN(0, 25, 1, 0x254, 0x5D0),
	MX53_PIN_FEC_RXD1 = _MXC_BUILD_GPIO_PIN(0, 26, 1, 0x258, 0x5D4),
	MX53_PIN_FEC_RXD0 = _MXC_BUILD_GPIO_PIN(0, 27, 1, 0x25C, 0x5D8),
	MX53_PIN_FEC_TX_EN = _MXC_BUILD_GPIO_PIN(0, 28, 1, 0x260, 0x5DC),
	MX53_PIN_FEC_TXD1 = _MXC_BUILD_GPIO_PIN(0, 29, 1, 0x264, 0x5E0),
	MX53_PIN_FEC_TXD0 = _MXC_BUILD_GPIO_PIN(0, 30, 1, 0x268, 0x5E4),
	MX53_PIN_FEC_MDC = _MXC_BUILD_GPIO_PIN(0, 31, 1, 0x26C, 0x5E8),
	MX53_PIN_NVCC_FEC = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x5EC),
	MX53_PIN_ATA_DIOW = _MXC_BUILD_GPIO_PIN(5, 17, 1, 0x270, 0x5F0),
	MX53_PIN_ATA_DMACK = _MXC_BUILD_GPIO_PIN(5, 18, 1, 0x274, 0x5F4),
	MX53_PIN_ATA_DMARQ = _MXC_BUILD_GPIO_PIN(6, 0, 1, 0x278, 0x5F8),
	MX53_PIN_ATA_BUFFER_EN = _MXC_BUILD_GPIO_PIN(6, 1, 1, 0x27C, 0x5FC),
	MX53_PIN_ATA_INTRQ = _MXC_BUILD_GPIO_PIN(6, 2, 1, 0x280, 0x600),
	MX53_PIN_ATA_DIOR = _MXC_BUILD_GPIO_PIN(6, 3, 1, 0x284, 0x604),
	MX53_PIN_ATA_RESET_B = _MXC_BUILD_GPIO_PIN(6, 4, 1, 0x288, 0x608),
	MX53_PIN_ATA_IORDY = _MXC_BUILD_GPIO_PIN(6, 5, 1, 0x28C, 0x60C),
	MX53_PIN_ATA_DA_0 = _MXC_BUILD_GPIO_PIN(6, 6, 1, 0x290, 0x610),
	MX53_PIN_ATA_DA_1 = _MXC_BUILD_GPIO_PIN(6, 7, 1, 0x294, 0x614),
	MX53_PIN_ATA_DA_2 = _MXC_BUILD_GPIO_PIN(6, 8, 1, 0x298, 0x618),
	MX53_PIN_ATA_CS_0 = _MXC_BUILD_GPIO_PIN(6, 9, 1, 0x29C, 0x61C),
	MX53_PIN_ATA_CS_1 = _MXC_BUILD_GPIO_PIN(6, 10, 1, 0x2A0, 0x620),
	MX53_PIN_NVCC_ATA2 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x624),
	MX53_PIN_ATA_DATA0 = _MXC_BUILD_GPIO_PIN(1, 0, 1, 0x2A4, 0x628),
	MX53_PIN_ATA_DATA1 = _MXC_BUILD_GPIO_PIN(1, 1, 1, 0x2A8, 0x62C),
	MX53_PIN_ATA_DATA2 = _MXC_BUILD_GPIO_PIN(1, 2, 1, 0x2AC, 0x630),
	MX53_PIN_ATA_DATA3 = _MXC_BUILD_GPIO_PIN(1, 3, 1, 0x2B0, 0x634),
	MX53_PIN_ATA_DATA4 = _MXC_BUILD_GPIO_PIN(1, 4, 1, 0x2B4, 0x638),
	MX53_PIN_ATA_DATA5 = _MXC_BUILD_GPIO_PIN(1, 5, 1, 0x2B8, 0x63C),
	MX53_PIN_ATA_DATA6 = _MXC_BUILD_GPIO_PIN(1, 6, 1, 0x2BC, 0x640),
	MX53_PIN_ATA_DATA7 = _MXC_BUILD_GPIO_PIN(1, 7, 1, 0x2C0, 0x644),
	MX53_PIN_ATA_DATA8 = _MXC_BUILD_GPIO_PIN(1, 8, 1, 0x2C4, 0x648),
	MX53_PIN_ATA_DATA9 = _MXC_BUILD_GPIO_PIN(1, 9, 1, 0x2C8, 0x64C),
	MX53_PIN_ATA_DATA10 = _MXC_BUILD_GPIO_PIN(1, 10, 1, 0x2CC, 0x650),
	MX53_PIN_ATA_DATA11 = _MXC_BUILD_GPIO_PIN(1, 11, 1, 0x2D0, 0x654),
	MX53_PIN_ATA_DATA12 = _MXC_BUILD_GPIO_PIN(1, 12, 1, 0x2D4, 0x658),
	MX53_PIN_ATA_DATA13 = _MXC_BUILD_GPIO_PIN(1, 13, 1, 0x2D8, 0x65C),
	MX53_PIN_ATA_DATA14 = _MXC_BUILD_GPIO_PIN(1, 14, 1, 0x2DC, 0x660),
	MX53_PIN_ATA_DATA15 = _MXC_BUILD_GPIO_PIN(1, 15, 1, 0x2E0, 0x664),
	MX53_PIN_NVCC_ATA0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x668),
	MX53_PIN_SD1_DATA0 = _MXC_BUILD_GPIO_PIN(0, 16, 1, 0x2E4, 0x66C),
	MX53_PIN_SD1_DATA1 = _MXC_BUILD_GPIO_PIN(0, 17, 1, 0x2E8, 0x670),
	MX53_PIN_SD1_CMD = _MXC_BUILD_GPIO_PIN(0, 18, 1, 0x2EC, 0x674),
	MX53_PIN_SD1_DATA2 = _MXC_BUILD_GPIO_PIN(0, 19, 1, 0x2F0, 0x678),
	MX53_PIN_SD1_CLK = _MXC_BUILD_GPIO_PIN(0, 20, 1, 0x2F4, 0x67C),
	MX53_PIN_SD1_DATA3 = _MXC_BUILD_GPIO_PIN(0, 21, 1, 0x2F8, 0x680),
	MX53_PIN_NVCC_SD1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x684),
	MX53_PIN_SD2_CLK = _MXC_BUILD_GPIO_PIN(0, 10, 1, 0x2FC, 0x688),
	MX53_PIN_SD2_CMD = _MXC_BUILD_GPIO_PIN(0, 11, 1, 0x300, 0x68C),
	MX53_PIN_SD2_DATA3 = _MXC_BUILD_GPIO_PIN(0, 12, 1, 0x304, 0x690),
	MX53_PIN_SD2_DATA2 = _MXC_BUILD_GPIO_PIN(0, 13, 1, 0x308, 0x694),
	MX53_PIN_SD2_DATA1 = _MXC_BUILD_GPIO_PIN(0, 14, 1, 0x30C, 0x698),
	MX53_PIN_SD2_DATA0 = _MXC_BUILD_GPIO_PIN(0, 15, 1, 0x310, 0x69C),
	MX53_PIN_NVCC_SD2 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6A0),
	MX53_PIN_GPIO_0 = _MXC_BUILD_GPIO_PIN(0, 0, 1, 0x314, 0x6A4),
	MX53_PIN_GPIO_1 = _MXC_BUILD_GPIO_PIN(0, 1, 1, 0x318, 0x6A8),
	MX53_PIN_GPIO_9 = _MXC_BUILD_GPIO_PIN(0, 9, 1, 0x31C, 0x6AC),
	MX53_PIN_GPIO_3 = _MXC_BUILD_GPIO_PIN(0, 3, 1, 0x320, 0x6B0),
	MX53_PIN_GPIO_6 = _MXC_BUILD_GPIO_PIN(0, 6, 1, 0x324, 0x6B4),
	MX53_PIN_GPIO_2 = _MXC_BUILD_GPIO_PIN(0, 2, 1, 0x328, 0x6B8),
	MX53_PIN_GPIO_4 = _MXC_BUILD_GPIO_PIN(0, 4, 1, 0x32C, 0x6BC),
	MX53_PIN_GPIO_5 = _MXC_BUILD_GPIO_PIN(0, 5, 1, 0x330, 0x6C0),
	MX53_PIN_GPIO_7 = _MXC_BUILD_GPIO_PIN(0, 7, 1, 0x334, 0x6C4),
	MX53_PIN_GPIO_8 = _MXC_BUILD_GPIO_PIN(0, 8, 1, 0x338, 0x6C8),
	MX53_PIN_GPIO_16 = _MXC_BUILD_GPIO_PIN(6, 11, 1, 0x33C, 0x6CC),
	MX53_PIN_GPIO_17 = _MXC_BUILD_GPIO_PIN(6, 12, 1, 0x340, 0x6D0),
	MX53_PIN_GPIO_18 = _MXC_BUILD_GPIO_PIN(6, 13, 1, 0x344, 0x6D4),
	MX53_PIN_NVCC_GPIO = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6D8),
	MX53_PIN_POR_B = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6DC),
	MX53_PIN_BOOT_MODE1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6E0),
	MX53_PIN_RESET_IN_B = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6E4),
	MX53_PIN_BOOT_MODE0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6E8),
	MX53_PIN_TEST_MODE = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6EC),
	MX53_PIN_GRP_ADDDS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6F0),
	MX53_PIN_GRP_DDRMODE_CTL = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6F4),
	MX53_PIN_GRP_DDRPKE = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x6FC),
	MX53_PIN_GRP_DDRPK = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x708),
	MX53_PIN_GRP_TERM_CTL3 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x70C),
	MX53_PIN_GRP_DDRHYS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x710),
	MX53_PIN_GRP_DDRMODE = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x714),
	MX53_PIN_GRP_B0DS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x718),
	MX53_PIN_GRP_B1DS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x71C),
	MX53_PIN_GRP_CTLDS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x720),
	MX53_PIN_GRP_DDR_TYPE = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x724),
	MX53_PIN_GRP_B2DS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x728),
	MX53_PIN_GRP_B3DS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x72C),
};

#endif				/* __ASSEMBLY__ */
#endif				/* __ASM_ARCH_MXC_MX53_PINS_H__ */
