/*
 * Copyright (C) 2010 Freescale Semiconductor, Inc. All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.

 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.

 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */
#ifndef __ASM_ARCH_MXC_MX50_PINS_H__
#define __ASM_ARCH_MXC_MX50_PINS_H__

#ifndef __ASSEMBLY__

/*
 * In order to identify pins more effectively, each mux-controlled pin's
 * enumerated value is constructed in the following way:
 *
 * -------------------------------------------------------------------
 * 31-29 | 28 - 24 |  23 - 21 | 20  - 10| 9 - 0
 * -------------------------------------------------------------------
 * IO_P  |  IO_I  | GPIO_I | PAD_I  | MUX_I
 * -------------------------------------------------------------------
 *
 * Bit 0 to 9 contains MUX_I used to identify the register
 * offset (0-based. base is IOMUX_module_base) defined in the Section
 * "sw_pad_ctl & sw_mux_ctl details" of the IC Spec. The
 * similar field definitions are used for the pad control register.
 * For example, the MX50_PIN_GPIO_19 is defined in the enumeration:
 *    ( (0x20 - MUX_I_START) << MUX_I)|( (0x348 - PAD_I_START) << PAD_I)
 * It means the mux control register is at register offset 0x20. The pad control
 * register offset is: 0x348 and also occupy the least significant bits
 * within the register.
 */

/*
 * Starting bit position within each entry of \b iomux_pins to represent the
 * MUX control register offset
 */
#define MUX_I			0
/*
 * Starting bit position within each entry of \b iomux_pins to represent the
 * PAD control register offset
 */
#define PAD_I			10
/*
 * Starting bit position within each entry of \b iomux_pins to represent which
 * mux mode is for GPIO (0-based)
 */
#define GPIO_I			21

#define MUX_IO_P                29
#define MUX_IO_I                24

#define NON_GPIO_PORT		0x7
#define PIN_TO_MUX_MASK		((1 << (PAD_I - MUX_I)) - 1)
#define PIN_TO_PAD_MASK		((1 << (GPIO_I - PAD_I)) - 1)
#define PIN_TO_ALT_GPIO_MASK	((1 << (MUX_IO_I - GPIO_I)) - 1)

#define NON_MUX_I		PIN_TO_MUX_MASK
#define NON_PAD_I		PIN_TO_PAD_MASK
#define MUX_I_START		0x0020
#define PAD_I_START		0x2CC
#define INPUT_CTL_START		0x6C4
#define MUX_I_END		(PAD_I_START - 4)

#define _MXC_BUILD_PIN(gp, gi, ga, mi, pi) \
	(((gp) << MUX_IO_P) | ((gi) << MUX_IO_I) | \
	((mi) << MUX_I) | \
	((pi - PAD_I_START) << PAD_I) | \
	((ga) << GPIO_I))

#define _MXC_BUILD_GPIO_PIN(gp, gi, ga, mi, pi) \
    _MXC_BUILD_PIN(gp, gi, ga, mi, pi)

#define _MXC_BUILD_NON_GPIO_PIN(mi, pi) \
    _MXC_BUILD_PIN(NON_GPIO_PORT, 0, 0, mi, pi)

#define PIN_TO_IOMUX_MUX(pin)	((pin >> MUX_I) & PIN_TO_MUX_MASK)
#define PIN_TO_IOMUX_PAD(pin)	((pin >> PAD_I) & PIN_TO_PAD_MASK)
#define PIN_TO_ALT_GPIO(pin)	((pin >> GPIO_I) & PIN_TO_ALT_GPIO_MASK)
#define PIN_TO_IOMUX_INDEX(pin)	(PIN_TO_IOMUX_MUX(pin) >> 2)

/*
 * This enumeration is constructed based on the Section
 * "sw_pad_ctl & sw_mux_ctl details" of the MX50 IC Spec. Each enumerated
 * value is constructed based on the rules described above.
 */
enum iomux_pins {
	MX50_PIN_KEY_COL0 = _MXC_BUILD_GPIO_PIN(3, 0, 1, 0x20, 0x2CC),
	MX50_PIN_KEY_ROW0 = _MXC_BUILD_GPIO_PIN(3, 1, 1, 0x24, 0x2D0),
	MX50_PIN_KEY_COL1 = _MXC_BUILD_GPIO_PIN(3, 2, 1, 0x28, 0x2D4),
	MX50_PIN_KEY_ROW1 = _MXC_BUILD_GPIO_PIN(3, 3, 1, 0x2C, 0x2D8),
	MX50_PIN_KEY_COL2 = _MXC_BUILD_GPIO_PIN(3, 4, 1, 0x30, 0x2DC),
	MX50_PIN_KEY_ROW2 = _MXC_BUILD_GPIO_PIN(3, 5, 1, 0x34, 0x2E0),
	MX50_PIN_KEY_COL3 = _MXC_BUILD_GPIO_PIN(3, 6, 1, 0x38, 0x2E4),
	MX50_PIN_KEY_ROW3 = _MXC_BUILD_GPIO_PIN(3, 7, 1, 0x3C, 0x2E8),
	MX50_PIN_I2C1_SCL = _MXC_BUILD_GPIO_PIN(5, 18, 1, 0x40, 0x2EC),
	MX50_PIN_I2C1_SDA = _MXC_BUILD_GPIO_PIN(5, 19, 1, 0x44, 0x2F0),
	MX50_PIN_I2C2_SCL = _MXC_BUILD_GPIO_PIN(5, 20, 1, 0x48, 0x2F4),
	MX50_PIN_I2C2_SDA = _MXC_BUILD_GPIO_PIN(5, 21, 1, 0x4C, 0x2F8),
	MX50_PIN_I2C3_SCL = _MXC_BUILD_GPIO_PIN(5, 22, 1, 0x50, 0x2FC),
	MX50_PIN_I2C3_SDA = _MXC_BUILD_GPIO_PIN(5, 23, 1, 0x54, 0x300),
	MX50_PIN_PWM1 = _MXC_BUILD_GPIO_PIN(5, 24, 1, 0x58, 0x304),
	MX50_PIN_PWM2 = _MXC_BUILD_GPIO_PIN(5, 25, 1, 0x5C, 0x308),
	MX50_PIN_OWIRE = _MXC_BUILD_GPIO_PIN(5, 26, 1, 0x60, 0x30C),
	MX50_PIN_EPITO = _MXC_BUILD_GPIO_PIN(5, 27, 1, 0x64, 0x310),
	MX50_PIN_WDOG = _MXC_BUILD_GPIO_PIN(5, 28, 1, 0x68, 0x314),
	MX50_PIN_SSI_TXFS = _MXC_BUILD_GPIO_PIN(5, 0, 1, 0x6C, 0x318),
	MX50_PIN_SSI_TXC = _MXC_BUILD_GPIO_PIN(5, 1, 1, 0x70, 0x31C),
	MX50_PIN_SSI_TXD = _MXC_BUILD_GPIO_PIN(5, 2, 1, 0x74, 0x320),
	MX50_PIN_SSI_RXD = _MXC_BUILD_GPIO_PIN(5, 3, 1, 0x78, 0x324),
	MX50_PIN_SSI_RXFS = _MXC_BUILD_GPIO_PIN(5, 4, 1, 0x7C, 0x328),
	MX50_PIN_SSI_RXC = _MXC_BUILD_GPIO_PIN(5, 5, 1, 0x80, 0x32C),
	MX50_PIN_UART1_TXD = _MXC_BUILD_GPIO_PIN(5, 6, 1, 0x84, 0x330),
	MX50_PIN_UART1_RXD = _MXC_BUILD_GPIO_PIN(5, 7, 1, 0x88, 0x334),
	MX50_PIN_UART1_CTS = _MXC_BUILD_GPIO_PIN(5, 8, 1, 0x8C, 0x338),
	MX50_PIN_UART1_RTS = _MXC_BUILD_GPIO_PIN(5, 9, 1, 0x90, 0x33C),
	MX50_PIN_UART2_TXD = _MXC_BUILD_GPIO_PIN(5, 10, 1, 0x94, 0x340),
	MX50_PIN_UART2_RXD = _MXC_BUILD_GPIO_PIN(5, 11, 1, 0x98, 0x344),
	MX50_PIN_UART2_CTS = _MXC_BUILD_GPIO_PIN(5, 12, 1, 0x9C, 0x348),
	MX50_PIN_UART2_RTS = _MXC_BUILD_GPIO_PIN(5, 13, 1, 0xA0, 0x34C),
	MX50_PIN_UART3_TXD = _MXC_BUILD_GPIO_PIN(5, 14, 1, 0xA4, 0x350),
	MX50_PIN_UART3_RXD = _MXC_BUILD_GPIO_PIN(5, 15, 1, 0xA8, 0x354),
	MX50_PIN_UART4_TXD = _MXC_BUILD_GPIO_PIN(5, 16, 1, 0xAC, 0x358),
	MX50_PIN_UART4_RXD = _MXC_BUILD_GPIO_PIN(5, 17, 1, 0xB0, 0x35C),
	MX50_PIN_CSPI_SCLK = _MXC_BUILD_GPIO_PIN(3, 8, 1, 0xB4, 0x360),
	MX50_PIN_CSPI_MOSI = _MXC_BUILD_GPIO_PIN(3, 9, 1, 0xB8, 0x364),
	MX50_PIN_CSPI_MISO = _MXC_BUILD_GPIO_PIN(3, 10, 1, 0xBC, 0x368),
	MX50_PIN_CSPI_SS0 = _MXC_BUILD_GPIO_PIN(3, 11, 1, 0xC0, 0x36C),
	MX50_PIN_ECSPI1_SCLK = _MXC_BUILD_GPIO_PIN(3, 12, 1, 0xC4, 0x370),
	MX50_PIN_ECSPI1_MOSI = _MXC_BUILD_GPIO_PIN(3, 13, 1, 0xC8, 0x374),
	MX50_PIN_ECSPI1_MISO = _MXC_BUILD_GPIO_PIN(3, 14, 1, 0xCC, 0x378),
	MX50_PIN_ECSPI1_SS0 = _MXC_BUILD_GPIO_PIN(3, 15, 1, 0xD0, 0x37C),
	MX50_PIN_ECSPI2_SCLK = _MXC_BUILD_GPIO_PIN(3, 16, 1, 0xD4, 0x380),
	MX50_PIN_ECSPI2_MOSI = _MXC_BUILD_GPIO_PIN(3, 17, 1, 0xD8, 0x384),
	MX50_PIN_ECSPI2_MISO = _MXC_BUILD_GPIO_PIN(3, 18, 1, 0xDC, 0x388),
	MX50_PIN_ECSPI2_SS0 = _MXC_BUILD_GPIO_PIN(3, 19, 1, 0xE0, 0x38C),
	MX50_PIN_SD1_CLK = _MXC_BUILD_GPIO_PIN(4, 0, 1, 0xE4, 0x390),
	MX50_PIN_SD1_CMD = _MXC_BUILD_GPIO_PIN(4, 1, 1, 0xE8, 0x394),
	MX50_PIN_SD1_D0 = _MXC_BUILD_GPIO_PIN(4, 2, 1, 0xEC, 0x398),
	MX50_PIN_SD1_D1 = _MXC_BUILD_GPIO_PIN(4, 3, 1, 0xF0, 0x39C),
	MX50_PIN_SD1_D2 = _MXC_BUILD_GPIO_PIN(4, 4, 1, 0xF4, 0x3A0),
	MX50_PIN_SD1_D3 = _MXC_BUILD_GPIO_PIN(4, 5, 1, 0xF8, 0x3A4),
	MX50_PIN_SD2_CLK = _MXC_BUILD_GPIO_PIN(4, 6, 1, 0xFC, 0x3A8),
	MX50_PIN_SD2_CMD = _MXC_BUILD_GPIO_PIN(4, 7, 1, 0x100, 0x3AC),
	MX50_PIN_SD2_D0 = _MXC_BUILD_GPIO_PIN(4, 8, 1, 0x104, 0x3B0),
	MX50_PIN_SD2_D1 = _MXC_BUILD_GPIO_PIN(4, 9, 1, 0x108, 0x3B4),
	MX50_PIN_SD2_D2 = _MXC_BUILD_GPIO_PIN(4, 10, 1, 0x10C, 0x3B8),
	MX50_PIN_SD2_D3 = _MXC_BUILD_GPIO_PIN(4, 11, 1, 0x110, 0x3BC),
	MX50_PIN_SD2_D4 = _MXC_BUILD_GPIO_PIN(4, 12, 1, 0x114, 0x3C0),
	MX50_PIN_SD2_D5 = _MXC_BUILD_GPIO_PIN(4, 13, 1, 0x118, 0x3C4),
	MX50_PIN_SD2_D6 = _MXC_BUILD_GPIO_PIN(4, 14, 1, 0x11C, 0x3C8),
	MX50_PIN_SD2_D7 = _MXC_BUILD_GPIO_PIN(4, 15, 1, 0x120, 0x3CC),
	MX50_PIN_SD2_WP = _MXC_BUILD_GPIO_PIN(4, 16, 1, 0x124, 0x3D0),
	MX50_PIN_SD2_CD = _MXC_BUILD_GPIO_PIN(4, 17, 1, 0x128, 0x3D4),
	MX50_PIN_PMIC_ON_REQ = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3D8),
	MX50_PIN_PMIC_STBY_REQ = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3DC),
	MX50_PIN_PMIC_PORT_B = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3E0),
	MX50_PIN_PMIC_BOOT_MODE1 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3E4),
	MX50_PIN_PMIC_RESET_IN_B = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3E8),
	MX50_PIN_PMIC_BOOT_MODE0 = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3EC),
	MX50_PIN_PMIC_TEST_MODE = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3F0),
	MX50_PIN_PMIC_JTAG_TMS = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3F4),
	MX50_PIN_PMIC_JTAG_MOD = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3F8),
	MX50_PIN_PMIC_JTAG_TRSTB = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x3FC),
	MX50_PIN_PMIC_JTAG_TDI = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x400),
	MX50_PIN_PMIC_JTAG_TCK = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x404),
	MX50_PIN_PMIC_JTAG_TDO = _MXC_BUILD_NON_GPIO_PIN(NON_MUX_I, 0x408),
	MX50_PIN_DISP_D0 = _MXC_BUILD_GPIO_PIN(1, 0, 1, 0x12C, 0x40C),
	MX50_PIN_DISP_D1 = _MXC_BUILD_GPIO_PIN(1, 1, 1, 0x130, 0x410),
	MX50_PIN_DISP_D2 = _MXC_BUILD_GPIO_PIN(1, 2, 1, 0x134, 0x414),
	MX50_PIN_DISP_D3 = _MXC_BUILD_GPIO_PIN(1, 3, 1, 0x138, 0x418),
	MX50_PIN_DISP_D4 = _MXC_BUILD_GPIO_PIN(1, 4, 1, 0x13C, 0x41C),
	MX50_PIN_DISP_D5 = _MXC_BUILD_GPIO_PIN(1, 5, 1, 0x140, 0x420),
	MX50_PIN_DISP_D6 = _MXC_BUILD_GPIO_PIN(1, 6, 1, 0x144, 0x424),
	MX50_PIN_DISP_D7 = _MXC_BUILD_GPIO_PIN(1, 7, 1, 0x148, 0x428),
	MX50_PIN_DISP_WR = _MXC_BUILD_GPIO_PIN(1, 16, 1, 0x14C, 0x42C),
	MX50_PIN_DISP_RD = _MXC_BUILD_GPIO_PIN(1, 19, 1, 0x150, 0x430),
	MX50_PIN_DISP_RS = _MXC_BUILD_GPIO_PIN(1, 17, 1, 0x154, 0x434),
	MX50_PIN_DISP_CS = _MXC_BUILD_GPIO_PIN(1, 21, 1, 0x158, 0x438),
	MX50_PIN_DISP_BUSY = _MXC_BUILD_GPIO_PIN(1, 18, 1, 0x15C, 0x43C),
	MX50_PIN_DISP_RESET = _MXC_BUILD_GPIO_PIN(1, 20, 1, 0x160, 0x440),
	MX50_PIN_SD3_CMD = _MXC_BUILD_GPIO_PIN(4, 18, 1, 0x164, 0x444),
	MX50_PIN_SD3_CLK = _MXC_BUILD_GPIO_PIN(4, 19, 1, 0x168, 0x448),
	MX50_PIN_SD3_D0 = _MXC_BUILD_GPIO_PIN(4, 20, 1, 0x16C, 0x44C),
	MX50_PIN_SD3_D1 = _MXC_BUILD_GPIO_PIN(4, 21, 1, 0x170, 0x450),
	MX50_PIN_SD3_D2 = _MXC_BUILD_GPIO_PIN(4, 22, 1, 0x174, 0x454),
	MX50_PIN_SD3_D3 = _MXC_BUILD_GPIO_PIN(4, 23, 1, 0x178, 0x458),
	MX50_PIN_SD3_D4 = _MXC_BUILD_GPIO_PIN(4, 24, 1, 0x17C, 0x45C),
	MX50_PIN_SD3_D5 = _MXC_BUILD_GPIO_PIN(4, 25, 1, 0x180, 0x460),
	MX50_PIN_SD3_D6 = _MXC_BUILD_GPIO_PIN(4, 26, 1, 0x184, 0x464),
	MX50_PIN_SD3_D7 = _MXC_BUILD_GPIO_PIN(4, 27, 1, 0x188, 0x468),
	MX50_PIN_SD3_WP = _MXC_BUILD_GPIO_PIN(4, 28, 1, 0x18C, 0x46C),
	MX50_PIN_DISP_D8 = _MXC_BUILD_GPIO_PIN(1, 8, 1, 0x190, 0x470),
	MX50_PIN_DISP_D9 = _MXC_BUILD_GPIO_PIN(1, 9, 1, 0x194, 0x474),
	MX50_PIN_DISP_D10 = _MXC_BUILD_GPIO_PIN(1, 10, 1, 0x198, 0x478),
	MX50_PIN_DISP_D11 = _MXC_BUILD_GPIO_PIN(1, 11, 1, 0x19C, 0x47C),
	MX50_PIN_DISP_D12 = _MXC_BUILD_GPIO_PIN(1, 12, 1, 0x1A0, 0x480),
	MX50_PIN_DISP_D13 = _MXC_BUILD_GPIO_PIN(1, 13, 1, 0x1A4, 0x484),
	MX50_PIN_DISP_D14 = _MXC_BUILD_GPIO_PIN(1, 14, 1, 0x1A8, 0x488),
	MX50_PIN_DISP_D15 = _MXC_BUILD_GPIO_PIN(1, 15, 1, 0x1AC, 0x48C),
	MX50_PIN_EPDC_D0 = _MXC_BUILD_GPIO_PIN(2, 0, 1, 0x1B0, 0x54C),
	MX50_PIN_EPDC_D1 = _MXC_BUILD_GPIO_PIN(2, 1, 1, 0x1B4, 0x550),
	MX50_PIN_EPDC_D2 = _MXC_BUILD_GPIO_PIN(2, 2, 1, 0x1B8, 0x554),
	MX50_PIN_EPDC_D3 = _MXC_BUILD_GPIO_PIN(2, 3, 1, 0x1BC, 0x558),
	MX50_PIN_EPDC_D4 = _MXC_BUILD_GPIO_PIN(2, 4, 1, 0x1C0, 0x55C),
	MX50_PIN_EPDC_D5 = _MXC_BUILD_GPIO_PIN(2, 5, 1, 0x1C4, 0x560),
	MX50_PIN_EPDC_D6 = _MXC_BUILD_GPIO_PIN(2, 6, 1, 0x1C8, 0x564),
	MX50_PIN_EPDC_D7 = _MXC_BUILD_GPIO_PIN(2, 7, 1, 0x1CC, 0x568),
	MX50_PIN_EPDC_D8 = _MXC_BUILD_GPIO_PIN(2, 8, 1, 0x1D0, 0x56C),
	MX50_PIN_EPDC_D9 = _MXC_BUILD_GPIO_PIN(2, 9, 1, 0x1D4, 0x570),
	MX50_PIN_EPDC_D10 = _MXC_BUILD_GPIO_PIN(2, 10, 1, 0x1D8, 0x574),
	MX50_PIN_EPDC_D11 = _MXC_BUILD_GPIO_PIN(2, 11, 1, 0x1DC, 0x578),
	MX50_PIN_EPDC_D12 = _MXC_BUILD_GPIO_PIN(2, 12, 1, 0x1E0, 0x57C),
	MX50_PIN_EPDC_D13 = _MXC_BUILD_GPIO_PIN(2, 13, 1, 0x1E4, 0x580),
	MX50_PIN_EPDC_D14 = _MXC_BUILD_GPIO_PIN(2, 14, 1, 0x1E8, 0x584),
	MX50_PIN_EPDC_D15 = _MXC_BUILD_GPIO_PIN(2, 15, 1, 0x1EC, 0x588),
	MX50_PIN_EPDC_GDCLK = _MXC_BUILD_GPIO_PIN(2, 16, 1, 0x1F0, 0x58C),
	MX50_PIN_EPDC_GDSP = _MXC_BUILD_GPIO_PIN(2, 17, 1, 0x1F4, 0x590),
	MX50_PIN_EPDC_GDOE = _MXC_BUILD_GPIO_PIN(2, 18, 1, 0x1F8, 0x594),
	MX50_PIN_EPDC_GDRL = _MXC_BUILD_GPIO_PIN(2, 19, 1, 0x1FC, 0x598),
	MX50_PIN_EPDC_SDCLK = _MXC_BUILD_GPIO_PIN(2, 20, 1, 0x200, 0x59C),
	MX50_PIN_EPDC_SDOEZ = _MXC_BUILD_GPIO_PIN(2, 21, 1, 0x204, 0x5A0),
	MX50_PIN_EPDC_SDOED = _MXC_BUILD_GPIO_PIN(2, 22, 1, 0x208, 0x5A4),
	MX50_PIN_EPDC_SDOE = _MXC_BUILD_GPIO_PIN(2, 23, 1, 0x20C, 0x5A8),
	MX50_PIN_EPDC_SDLE = _MXC_BUILD_GPIO_PIN(2, 24, 1, 0x210, 0x5AC),
	MX50_PIN_EPDC_SDCLKN = _MXC_BUILD_GPIO_PIN(2, 25, 1, 0x214, 0x5B0),
	MX50_PIN_EPDC_SDSHR = _MXC_BUILD_GPIO_PIN(2, 26, 1, 0x218, 0x5B4),
	MX50_PIN_EPDC_PWRCOM = _MXC_BUILD_GPIO_PIN(2, 27, 1, 0x21C, 0x5B8),
	MX50_PIN_EPDC_PWRSTAT = _MXC_BUILD_GPIO_PIN(2, 28, 1, 0x220, 0x5BC),
	MX50_PIN_EPDC_PWRCTRL0 = _MXC_BUILD_GPIO_PIN(2, 29, 1, 0x224, 0x5C0),
	MX50_PIN_EPDC_PWRCTRL1 = _MXC_BUILD_GPIO_PIN(2, 30, 1, 0x228, 0x5C4),
	MX50_PIN_EPDC_PWRCTRL2 = _MXC_BUILD_GPIO_PIN(2, 31, 1, 0x22C, 0x5C8),
	MX50_PIN_EPDC_PWRCTRL3 = _MXC_BUILD_GPIO_PIN(3, 20, 1, 0x230, 0x5CC),
	MX50_PIN_EPDC_VCOM0 = _MXC_BUILD_GPIO_PIN(3, 21, 1, 0x234, 0x5D0),
	MX50_PIN_EPDC_VCOM1 = _MXC_BUILD_GPIO_PIN(3, 22, 1, 0x238, 0x5D4),
	MX50_PIN_EPDC_BDR0 = _MXC_BUILD_GPIO_PIN(3, 23, 1, 0x23C, 0x5D8),
	MX50_PIN_EPDC_BDR1 = _MXC_BUILD_GPIO_PIN(3, 24, 1, 0x240, 0x5DC),
	MX50_PIN_EPDC_SDCE0 = _MXC_BUILD_GPIO_PIN(3, 25, 1, 0x244, 0x5E0),
	MX50_PIN_EPDC_SDCE1 = _MXC_BUILD_GPIO_PIN(3, 26, 1, 0x248, 0x5E4),
	MX50_PIN_EPDC_SDCE2 = _MXC_BUILD_GPIO_PIN(3, 27, 1, 0x24C, 0x5E8),
	MX50_PIN_EPDC_SDCE3 = _MXC_BUILD_GPIO_PIN(3, 28, 1, 0x250, 0x5EC),
	MX50_PIN_EPDC_SDCE4 = _MXC_BUILD_GPIO_PIN(3, 29, 1, 0x254, 0x5F0),
	MX50_PIN_EPDC_SDCE5 = _MXC_BUILD_GPIO_PIN(3, 30, 1, 0x258, 0x5F4),
	MX50_PIN_EIM_DA0 = _MXC_BUILD_GPIO_PIN(0, 0, 1, 0x25C, 0x5F8),
	MX50_PIN_EIM_DA1 = _MXC_BUILD_GPIO_PIN(0, 1, 1, 0x260, 0x5FC),
	MX50_PIN_EIM_DA2 = _MXC_BUILD_GPIO_PIN(0, 2, 1, 0x264, 0x600),
	MX50_PIN_EIM_DA3 = _MXC_BUILD_GPIO_PIN(0, 3, 1, 0x268, 0x604),
	MX50_PIN_EIM_DA4 = _MXC_BUILD_GPIO_PIN(0, 4, 1, 0x26C, 0x608),
	MX50_PIN_EIM_DA5 = _MXC_BUILD_GPIO_PIN(0, 5, 1, 0x270, 0x60C),
	MX50_PIN_EIM_DA6 = _MXC_BUILD_GPIO_PIN(0, 6, 1, 0x274, 0x610),
	MX50_PIN_EIM_DA7 = _MXC_BUILD_GPIO_PIN(0, 7, 1, 0x278, 0x614),
	MX50_PIN_EIM_DA8 = _MXC_BUILD_GPIO_PIN(0, 8, 1, 0x27C, 0x618),
	MX50_PIN_EIM_DA9 = _MXC_BUILD_GPIO_PIN(0, 9, 1, 0x280, 0x61C),
	MX50_PIN_EIM_DA10 = _MXC_BUILD_GPIO_PIN(0, 10, 1, 0x284, 0x620),
	MX50_PIN_EIM_DA11 = _MXC_BUILD_GPIO_PIN(0, 11, 1, 0x288, 0x624),
	MX50_PIN_EIM_DA12 = _MXC_BUILD_GPIO_PIN(0, 12, 1, 0x28C, 0x628),
	MX50_PIN_EIM_DA13 = _MXC_BUILD_GPIO_PIN(0, 13, 1, 0x290, 0x62C),
	MX50_PIN_EIM_DA14 = _MXC_BUILD_GPIO_PIN(0, 14, 1, 0x294, 0x630),
	MX50_PIN_EIM_DA15 = _MXC_BUILD_GPIO_PIN(0, 15, 1, 0x298, 0x634),
	MX50_PIN_EIM_CS2 = _MXC_BUILD_GPIO_PIN(0, 16, 1, 0x29C, 0x638),
	MX50_PIN_EIM_CS1 = _MXC_BUILD_GPIO_PIN(0, 17, 1, 0x2A0, 0x63C),
	MX50_PIN_EIM_CS0 = _MXC_BUILD_GPIO_PIN(0, 18, 1, 0x2A4, 0x640),
	MX50_PIN_EIM_EB0 = _MXC_BUILD_GPIO_PIN(0, 19, 1, 0x2A8, 0x644),
	MX50_PIN_EIM_EB1 = _MXC_BUILD_GPIO_PIN(0, 20, 1, 0x2AC, 0x648),
	MX50_PIN_EIM_WAIT = _MXC_BUILD_GPIO_PIN(0, 21, 1, 0x2B0, 0x64C),
	MX50_PIN_EIM_BCLK = _MXC_BUILD_GPIO_PIN(0, 22, 1, 0x2B4, 0x650),
	MX50_PIN_EIM_RDY = _MXC_BUILD_GPIO_PIN(0, 23, 1, 0x2B8, 0x654),
	MX50_PIN_EIM_OE = _MXC_BUILD_GPIO_PIN(0, 24, 1, 0x2BC, 0x658),
	MX50_PIN_EIM_RW = _MXC_BUILD_GPIO_PIN(0, 25, 1, 0x2C0, 0x65C),
	MX50_PIN_EIM_LBA = _MXC_BUILD_GPIO_PIN(0, 26, 1, 0x2C4, 0x660),
	MX50_PIN_EIM_CRE = _MXC_BUILD_GPIO_PIN(0, 27, 1, 0x2C8, 0x664),
};

#endif				/* __ASSEMBLY__ */
#endif				/* __ASM_ARCH_MXC_MX50_PINS_H__ */
