/*
 * Copyright (C) 2012 Simon Budig <simon.budig@kernelconcepts.de>
 *
 * This code is based on Linux Kernel sourcecode:
 * Author: Juergen Beisert, Pengutronix
 * Author: Vitaly Wool <vital@embeddedalley.com>
 *
 * Copyright 2008-2009 Freescale Semiconductor, Inc. All Rights Reserved.
 * Copyright 2008 Embedded Alley Solutions, Inc All Rights Reserved.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 */

#include <common.h>
#include <lcd.h>
#include <asm/io.h>
#include <asm/arch/regs-clkctrl.h>

#define REG_SET_ADD 4
#define REG_CLR_ADD 8

#define LCDC_CTRL                       0x00
#define LCDC_CTRL1                      0x10
#define LCDC_TIMING                     0x60
#define LCDC_VDCTRL0                    0x70
#define LCDC_VDCTRL1                    0x80
#define LCDC_VDCTRL2                    0x90
#define LCDC_VDCTRL3                    0xa0
#define LCDC_VDCTRL4                    0xb0
#define LCDC_DVICTRL0                   0xc0
#define LCDC_DVICTRL1                   0xd0
#define LCDC_DVICTRL2                   0xe0
#define LCDC_DVICTRL3                   0xf0
#define LCDC_DVICTRL4                   0x100

#ifdef CONFIG_MX28
# define LCDC_BASE                      0x80030000 
# define LCDC_V4_CTRL2                  0x20
# define LCDC_TRANSFER_COUNT            0x30
# define LCDC_CUR_BUF                   0x40
# define LCDC_NEXT_BUF                  0x50
# define LCDC_DATA                      0x180
# define LCDC_DEBUG0                    0x1d0
# define VDCTRL2_SET_HSYNC_PULSE_WIDTH(x)  (((x) & 0x3fff) << 18)
# define VDCTRL2_GET_HSYNC_PULSE_WIDTH(x)  (((x) >> 18) & 0x3fff)
#else
# error MXS Framebuffer needs MX28 for now
# define LCDC_BASE                      ???
# define LCDC_TRANSFER_COUNT            0x20
# define LCDC_CUR_BUF                   0x30
# define LCDC_NEXT_BUF                  0x40
# define LCDC_DATA                      0x1b0
# define LCDC_DEBUG0                    0x1f0
# define VDCTRL2_SET_HSYNC_PULSE_WIDTH(x)  (((x) & 0xff) << 24)
# define VDCTRL2_GET_HSYNC_PULSE_WIDTH(x)  (((x) >> 24) & 0xff)
#endif

#define CTRL_SFTRST                     (1 << 31)
#define CTRL_CLKGATE                    (1 << 30)
#define CTRL_BYPASS_COUNT               (1 << 19)
#define CTRL_VSYNC_MODE                 (1 << 18)
#define CTRL_DOTCLK_MODE                (1 << 17)
#define CTRL_DATA_SELECT                (1 << 16)
#define CTRL_SET_BUS_WIDTH(x)           (((x) & 0x3) << 10)
#define CTRL_GET_BUS_WIDTH(x)           (((x) >> 10) & 0x3)
#define CTRL_SET_WORD_LENGTH(x)         (((x) & 0x3) << 8)
#define CTRL_GET_WORD_LENGTH(x)         (((x) >> 8) & 0x3)
#define CTRL_MASTER                     (1 << 5)
#define CTRL_DF16                       (1 << 3)
#define CTRL_DF18                       (1 << 2)
#define CTRL_DF24                       (1 << 1)
#define CTRL_RUN                        (1 << 0)

#define CTRL1_FIFO_CLEAR                (1 << 21)
#define CTRL1_SET_BYTE_PACKAGING(x)     (((x) & 0xf) << 16)
#define CTRL1_GET_BYTE_PACKAGING(x)     (((x) >> 16) & 0xf)

#define TRANSFER_COUNT_SET_VCOUNT(x)    (((x) & 0xffff) << 16)
#define TRANSFER_COUNT_GET_VCOUNT(x)    (((x) >> 16) & 0xffff)
#define TRANSFER_COUNT_SET_HCOUNT(x)    ((x) & 0xffff)
#define TRANSFER_COUNT_GET_HCOUNT(x)    ((x) & 0xffff)


#define VDCTRL0_ENABLE_PRESENT          (1 << 28)
#define VDCTRL0_VSYNC_ACT_HIGH          (1 << 27)
#define VDCTRL0_HSYNC_ACT_HIGH          (1 << 26)
#define VDCTRL0_DOTCLK_ACT_FAILING      (1 << 25)
#define VDCTRL0_ENABLE_ACT_HIGH         (1 << 24)
#define VDCTRL0_VSYNC_PERIOD_UNIT       (1 << 21)
#define VDCTRL0_VSYNC_PULSE_WIDTH_UNIT  (1 << 20)
#define VDCTRL0_HALF_LINE               (1 << 19)
#define VDCTRL0_HALF_LINE_MODE          (1 << 18)
#define VDCTRL0_SET_VSYNC_PULSE_WIDTH(x) ((x) & 0x3ffff)
#define VDCTRL0_GET_VSYNC_PULSE_WIDTH(x) ((x) & 0x3ffff)

#define VDCTRL2_SET_HSYNC_PERIOD(x)     ((x) & 0x3ffff)
#define VDCTRL2_GET_HSYNC_PERIOD(x)     ((x) & 0x3ffff)

#define VDCTRL3_MUX_SYNC_SIGNALS        (1 << 29)
#define VDCTRL3_VSYNC_ONLY              (1 << 28)
#define SET_HOR_WAIT_CNT(x)             (((x) & 0xfff) << 16)
#define GET_HOR_WAIT_CNT(x)             (((x) >> 16) & 0xfff)
#define SET_VERT_WAIT_CNT(x)            ((x) & 0xffff)
#define GET_VERT_WAIT_CNT(x)            ((x) & 0xffff)

#define VDCTRL4_SET_DOTCLK_DLY(x)       (((x) & 0x7) << 29) /* v4 only */
#define VDCTRL4_GET_DOTCLK_DLY(x)       (((x) >> 29) & 0x7) /* v4 only */
#define VDCTRL4_SYNC_SIGNALS_ON         (1 << 18)
#define SET_DOTCLK_H_VALID_DATA_CNT(x)  ((x) & 0x3ffff)

#define DEBUG0_HSYNC                    (1 < 26)
#define DEBUG0_VSYNC                    (1 < 25)

#define MIN_XRES                        120
#define MIN_YRES                        120

DECLARE_GLOBAL_DATA_PTR;

void *lcd_base;			/* Start of framebuffer memory	*/
void *lcd_console_address;	/* Start of console buffer	*/

int lcd_line_length;
int lcd_color_fg;
int lcd_color_bg;

short console_col;
short console_row;


void lcd_initcolregs(void)
{
}

void lcd_setcolreg(ushort regno, ushort red, ushort green, ushort blue)
{
}

void lcd_enable(void)
{
}

void lcd_disable(void)
{
}

void lcd_panel_disable(void)
{
}

void lcd_ctrl_init(void *lcdbase)
{
	u32 vdctrl0, vdctrl4, reg;

	/* enable clock */
	reg = readl(CLKCTRL_BASE_ADDR + HW_CLKCTRL_DIS_LCDIF);
	reg &= ~BM_CLKCTRL_DIS_LCDIF_CLKGATE;
	writel(reg, CLKCTRL_BASE_ADDR + HW_CLKCTRL_DIS_LCDIF);

	/* bring lcd block out of reset */
	mxs_reset_block (LCDC_BASE + LCDC_CTRL);

	/* clear the FIFOs */
	writel(CTRL1_FIFO_CLEAR, LCDC_BASE + LCDC_CTRL1 + REG_SET_ADD);

	/* 16 bit mode */
	writel(CTRL1_SET_BYTE_PACKAGING(0xf), LCDC_BASE + LCDC_CTRL1);
	writel(CTRL_BYPASS_COUNT | CTRL_MASTER | CTRL_SET_BUS_WIDTH(panel_info.ld_intf_width) | CTRL_SET_WORD_LENGTH(0), LCDC_BASE + LCDC_CTRL);

	writel(TRANSFER_COUNT_SET_VCOUNT(panel_info.vl_row) |
	       TRANSFER_COUNT_SET_HCOUNT(panel_info.vl_col),
	       LCDC_BASE + LCDC_TRANSFER_COUNT);

	vdctrl0 = VDCTRL0_ENABLE_PRESENT |      /* always in DOTCLOCK mode */
		VDCTRL0_VSYNC_PERIOD_UNIT |
		VDCTRL0_VSYNC_PULSE_WIDTH_UNIT |
		VDCTRL0_SET_VSYNC_PULSE_WIDTH(panel_info.vl_vsync);

	vdctrl0 |= panel_info.vl_sync;

	writel(vdctrl0, LCDC_BASE + LCDC_VDCTRL0);

	/* frame length in lines */
	writel(panel_info.vl_upper_margin + panel_info.vl_vsync +
		panel_info.vl_lower_margin + panel_info.vl_row,
		LCDC_BASE + LCDC_VDCTRL1);

	/* line length in units of clocks or pixels */
	writel(VDCTRL2_SET_HSYNC_PULSE_WIDTH(panel_info.vl_hsync) |
		VDCTRL2_SET_HSYNC_PERIOD(panel_info.vl_left_margin +
		panel_info.vl_hsync + panel_info.vl_right_margin +
		panel_info.vl_col),
		LCDC_BASE + LCDC_VDCTRL2);

	writel(SET_HOR_WAIT_CNT(panel_info.vl_left_margin +
		panel_info.vl_hsync) |
		SET_VERT_WAIT_CNT(panel_info.vl_upper_margin +
			panel_info.vl_vsync),
		LCDC_BASE + LCDC_VDCTRL3);

	vdctrl4 = SET_DOTCLK_H_VALID_DATA_CNT(panel_info.vl_col);
#ifdef CONFIG_MX28
	vdctrl4 |= VDCTRL4_SET_DOTCLK_DLY(panel_info.dotclk_delay);
#endif
	writel(vdctrl4, LCDC_BASE + LCDC_VDCTRL4);

	writel(gd->fb_base, LCDC_BASE + LCDC_NEXT_BUF);

#if 0
	/* change div */
	reg = readl(CLKCTRL_BASE_ADDR + HW_CLKCTRL_DIS_LCDIF);
	reg &= ~BM_CLKCTRL_DIS_LCDIF_DIV;
	reg |= (24000000/1000000000UL*(panel_info.vl_pixclock) * 1000U) << BP_CLKCTRL_DIS_LCDIF_DIV;
	writel(reg, CLKCTRL_BASE_ADDR + HW_CLKCTRL_DIS_LCDIF);
#endif

	writel(CTRL_DOTCLK_MODE, LCDC_BASE + LCDC_CTRL + REG_SET_ADD);

	/* enable the SYNC signals first, then the DMA engine */
	reg = readl(LCDC_BASE + LCDC_VDCTRL4);
	reg |= VDCTRL4_SYNC_SIGNALS_ON;
	writel(reg, LCDC_BASE + LCDC_VDCTRL4);

	writel(CTRL_RUN, LCDC_BASE + LCDC_CTRL + REG_SET_ADD);
}

ulong calc_fbsize(void)
{
	return panel_info.vl_row * panel_info.vl_col * 2 \
		* NBITS(panel_info.vl_bpix) / 8;
}


