/*
 * Copyright (C) 2011 Lothar Waßmann <LW@KARO-electronics.de>
 * based on: board/freesclae/mx28_evk.c (C) 2010 Freescale Semiconductor, Inc.
 *
 * See file CREDITS for list of people who contributed to this
 * project.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of
 * the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston,
 * MA 02111-1307 USA
 */

#include <common.h>
#include <asm/io.h>
#include <asm/arch/regs-pinctrl.h>
#include <asm/arch/pinctrl.h>
#include <asm/arch/regs-clkctrl.h>
#include <asm/arch/regs-ocotp.h>
#include <asm/errno.h>

#include <mmc.h>
#include <imx_ssp_mmc.h>

#include <lcd.h>
#include <spi.h>
#include <i2c.h>

#include <malloc.h>
#include <spi_flash.h>

#include "becker-logo.h"

/* This should be removed after it's added into mach-types.h */

static const int mach_type = MACH_TYPE_CC51;

DECLARE_GLOBAL_DATA_PTR;

#ifdef CONFIG_IMX_SSP_MMC

/* MMC pins */
static struct pin_desc mmc0_pins_desc[] = {
	{ PINID_SSP0_DATA0, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DATA1, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DATA2, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DATA3, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DATA4, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DATA5, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DATA6, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DATA7, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_CMD, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_DETECT, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP0_SCK, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
};

static struct pin_group mmc0_pins = {
	.pins		= mmc0_pins_desc,
	.nr_pins	= ARRAY_SIZE(mmc0_pins_desc)
};

struct imx_ssp_mmc_cfg ssp_mmc_cfg[2] = {
	{REGS_SSP0_BASE, HW_CLKCTRL_SSP0, BM_CLKCTRL_CLKSEQ_BYPASS_SSP0},
	{REGS_SSP1_BASE, HW_CLKCTRL_SSP1, BM_CLKCTRL_CLKSEQ_BYPASS_SSP1},
};
#endif

/* ENET pins */
static struct pin_desc enet_pins_desc[] = {
	{ PINID_ENET0_MDC, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET0_MDIO, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET0_RX_EN, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET0_RXD0, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET0_RXD1, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET0_TX_EN, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET0_TXD0, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET0_TXD1, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_ENET_CLK, PIN_FUN1, PAD_8MA, PAD_3V3, 1 }
};

static struct pin_group enet_pins = {
	.pins		= enet_pins_desc,
	.nr_pins	= ARRAY_SIZE(enet_pins_desc),
};

static struct pin_desc duart_pins_desc[] = {
	{ PINID_AUART0_RTS, PIN_FUN3, PAD_8MA, PAD_3V3, 1 },
	{ PINID_AUART0_CTS, PIN_FUN3, PAD_8MA, PAD_3V3, 1 },
	{ PINID_AUART0_TX, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
	{ PINID_AUART0_RX, PIN_FUN1, PAD_8MA, PAD_3V3, 1 },
};

static struct pin_group duart_pins = {
	.pins = duart_pins_desc,
	.nr_pins = ARRAY_SIZE(duart_pins_desc),
};

static struct pin_desc gpmi_pins_desc[] = {
	{ PINID_GPMI_D00,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_D01,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_D02,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_D03,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_D04,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_D05,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_D06,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_D07,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_CE0N, PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_RDY0, PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_RDN,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_WRN,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_ALE,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
	{ PINID_GPMI_CLE,  PIN_FUN1, PAD_4MA, PAD_3V3, 1},
};

static struct pin_group gpmi_pins = {
	.pins		= gpmi_pins_desc,
	.nr_pins	= ARRAY_SIZE(gpmi_pins_desc),
};

#ifdef CONFIG_LCD

static struct pin_desc lcd_pins_desc[] = {
	{ PINID_LCD_D00, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D01, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D02, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D03, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D04, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D05, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D06, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D07, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D08, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D09, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D10, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D11, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D12, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D13, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D14, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },
	{ PINID_LCD_D15, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },

	{ PINID_LCD_RD_E, PIN_FUN2, PAD_4MA, PAD_3V3, 0 },   /* LCD_VSYNC */
	{ PINID_LCD_WR_RWN, PIN_FUN2, PAD_4MA, PAD_3V3, 0 }, /* LCD_HSYNC */
	{ PINID_LCD_RS, PIN_FUN2, PAD_4MA, PAD_3V3, 0 },     /* LCD_DOTCLK */
	{ PINID_LCD_CS, PIN_FUN1, PAD_4MA, PAD_3V3, 0 },

	{ PINID_LCD_ENABLE, PIN_GPIO, PAD_8MA, PAD_3V3, 0 },  /* GPIO 1,31 */
	{ PINID_LCD_RESET, PIN_GPIO, PAD_8MA, PAD_3V3, 0 },   /* GPIO 3,30 */
	{ PINID_PWM1, PIN_FUN1, PAD_12MA, PAD_3V3, 0 },       /* GPIO 3,17 */
};

static struct pin_group lcd_pins = {
	.pins = lcd_pins_desc,
	.nr_pins = ARRAY_SIZE(lcd_pins_desc),
};

vidinfo_t panel_info = {
	vl_refresh: 60,
	vl_col: 480,
	vl_row: 272,
	vl_pixclock: 111111,
	vl_left_margin:  2,
	vl_right_margin: 40,
	vl_upper_margin: 10,
	vl_lower_margin: 1,
	vl_hsync: 42,
	vl_vsync: 2,
	vl_sync : 0,
	dotclk_delay: 2,
	ld_intf_width: 0,     /* STMLCDIF_16BIT */
	vl_bpix: 4,
};

#endif

static struct pin_desc spi2_pins_desc[] = {
	{ PINID_SSP2_SCK,  PIN_GPIO, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP2_MOSI, PIN_GPIO, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP2_MISO, PIN_GPIO, PAD_8MA, PAD_3V3, 1 },
	{ PINID_SSP2_SS0,  PIN_GPIO, PAD_8MA, PAD_3V3, 1 },
};

static struct pin_group spi2_pins = {
	.pins = spi2_pins_desc,
	.nr_pins = ARRAY_SIZE(spi2_pins_desc),
};

static struct pin_desc i2c_pins_desc[] = {
	{ PINID_I2C0_SCL, PIN_GPIO, PAD_8MA, PAD_3V3, 1 },
	{ PINID_I2C0_SDA, PIN_GPIO, PAD_8MA, PAD_3V3, 1 },
	{ PINID_GPMI_RESETN, PIN_GPIO, PAD_8MA, PAD_3V3, 1 },
};

static struct pin_group i2c_pins = {
	.pins = i2c_pins_desc,
	.nr_pins = ARRAY_SIZE(i2c_pins_desc),
};


/*
 * Functions
 */
static void duart_init(void)
{
	pin_set_group(&duart_pins);
}

#ifdef CONFIG_LCD

#define PWM_BASE      0x80064000
#define PWM_CTRL      0x00
#define PWM_CTRL_SET  0x04
#define PWM_ACTIVE1   0x30
#define PWM_PERIOD1   0x40

static int lcd_cc51_init (void)
{
	/* enable XTAL for PWM */
	writel (BM_CLKCTRL_XTAL_PWM_CLK24M_GATE, CLKCTRL_BASE_ADDR + HW_CLKCTRL_XTAL_CLR);

	/* enable PWM block */
	mxs_reset_block (PWM_BASE);

	/* backlight off */
	writel (0x00000002, PWM_BASE + PWM_CTRL_SET);
	writel (0x00000000, PWM_BASE + PWM_ACTIVE1);
	writel (0x000e00ff, PWM_BASE + PWM_PERIOD1);
	pin_gpio_direction(PINID_PWM1, 1);

	pin_set_group(&lcd_pins);

	pin_gpio_set (PINID_LCD_ENABLE, 1);
	pin_gpio_direction(PINID_LCD_ENABLE, 1);
	udelay (15);
	pin_gpio_set (PINID_LCD_RESET, 1);
	pin_gpio_direction(PINID_LCD_RESET, 1);
	udelay (200);
	// udelay (200);
	// udelay (200);
	// udelay (200);
	// udelay (200);

	gd->fb_base = CONFIG_FB_BASE;

	return 0;
}

#endif

static void spi2_cc51_init(void)
{
	pin_set_group(&spi2_pins);

	pin_gpio_set (PINID_SSP2_SCK, 1);
	pin_gpio_set (PINID_SSP2_MOSI, 1);
	pin_gpio_set (PINID_SSP2_SS0, 1);
	pin_gpio_direction (PINID_SSP2_SCK, 1);
	pin_gpio_direction (PINID_SSP2_MOSI, 1);
	pin_gpio_direction (PINID_SSP2_MISO, 0);
	pin_gpio_direction (PINID_SSP2_SS0, 1);
}

static void i2c_cc51_init(void)
{
	int i;
	pin_set_group(&i2c_pins);

	pin_gpio_set (PINID_I2C0_SCL, 1);
	pin_gpio_set (PINID_I2C0_SDA, 1);
	pin_gpio_set (PINID_GPMI_RESETN, 0);
	pin_gpio_direction (PINID_I2C0_SCL, 1);
	pin_gpio_direction (PINID_I2C0_SDA, 1);
	pin_gpio_direction (PINID_GPMI_RESETN, 1);
	/* allow 6ms for reset to work */
	for (i = 0; i < 30; i++)
		udelay (200);
	pin_gpio_set (PINID_GPMI_RESETN, 1);
	for (i = 0; i < 5; i++)
		udelay (200);
}

int board_init(void)
{
	int i, ret;
	uint8_t data[3];
	static const uint8_t smsc2517_initdata[] = {
		0x24,   /* 0x00 */
		0x04,
		0x17,
		0x25,
		0x00,
		0x00,
		0x9b,   /* 0b10011011 */
		0x20,

		0x0b,   /* 0x08 */
		0x00,
		0xc0,
		0xc0,
		0x01,
		0x32,
		0x01,
		0x32,
		0x32,
		0x00, 0x00,  /* Language ID */
		0x0f, /* Manufacturer String length */
		0x10, /* Product String Length */
		0x00, /* Serialno String Length */

		/* 0x16: Manufacturer String */
		'B',  0x00, 'e',  0x00, 'c',  0x00, 'k',  0x00,
		'e',  0x00, 'r',  0x00, '-',  0x00, 'A',  0x00,
		'n',  0x00, 't',  0x00, 'r',  0x00, 'i',  0x00,
		'e',  0x00, 'b',  0x00, 'e',  0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

		/* 0x54: Product String */
		'C',  0x00, 'C',  0x00, '5',  0x00, '1',  0x00,
		' ',  0x00, 'b',  0x00, 'u',  0x00, 'i',  0x00,
		'l',  0x00, 't',  0x00, 'i',  0x00, 'n',  0x00,
		' ',  0x00, 'H',  0x00, 'u',  0x00, 'b',  0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00,

		/* 0x92: Serial String */
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
		0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	};

	/* Hardware numbering on CC51:
	 *  4  1  3  2
	 *  5
	 * gets mapped to:
	 *  1  2  3  4
	 *  5
	 */
	static const uint8_t smsc2517_portmapdata[] = {
		/* 0xfb: Port Map 1-7 */
		0x42, 0x13, 0x05, 0x00,
	};

	gd->bd->bi_arch_number = mach_type;

	/* Address of boot parameters */
	gd->bd->bi_boot_params = PHYS_SDRAM_1 + 0x1000;

	duart_init();

#ifdef CONFIG_LCD
	lcd_cc51_init();
#endif
	spi2_cc51_init();
	i2c_cc51_init();

	/* set up the USB hub, up to 32 bytes/write allowed */
	for (i = 0; i < sizeof (smsc2517_initdata); i += 32) {
		/* two byte address encodes offset and length */
		ret = i2c_write(0x2c, 32 | (i << 8), 2,
		                smsc2517_initdata + i, 32);
	}

	ret = i2c_write(0x2c, 4 | (0xfb << 8), 2,
	                smsc2517_portmapdata, 4);
	udelay (200);

	/* enable USB host */
	data[0] = 0xff;
	data[1] = 0x01;
	data[2] = 0x01;
	ret = i2c_write(0x2c, 0x00, 0, data, 3) | ret;

	return 0;
}

int dram_init(void)
{
	gd->bd->bi_dram[0].start = PHYS_SDRAM_1;
	gd->bd->bi_dram[0].size = PHYS_SDRAM_1_SIZE;

	return 0;
}

#ifdef CONFIG_IMX_SSP_MMC

#ifdef CONFIG_DYNAMIC_MMC_DEVNO
int get_mmc_env_devno(void)
{
	unsigned long global_boot_mode;

	global_boot_mode = REG_RD_ADDR(GLOBAL_BOOT_MODE_ADDR);
	return ((global_boot_mode & 0xf) == BOOT_MODE_SD1) ? 1 : 0;
}
#endif


u32 ssp_mmc_is_wp(struct mmc *mmc)
{
	return 0;
}

int ssp_mmc_gpio_init(bd_t *bis)
{
	s32 status = 0;
	u32 index = 0;

	for (index = 0; index < CONFIG_SYS_SSP_MMC_NUM; index++) {
		switch (index) {
		case 0:
			/* Set up MMC pins */
			pin_set_group(&mmc0_pins);
			break;

		default:
			printf("Warning: more ssp mmc controllers configured(%d) than supported by the board(2)\n",
				CONFIG_SYS_SSP_MMC_NUM);
			return status;
		}
		status |= imx_ssp_mmc_initialize(bis, &ssp_mmc_cfg[index]);
	}

	return status;
}

int board_mmc_init(bd_t *bis)
{
	if (!ssp_mmc_gpio_init(bis))
		return 0;
	else
		return -1;
}

#endif

#if defined(CONFIG_MXC_FEC) && defined(CONFIG_GET_FEC_MAC_ADDR_FROM_IIM)
int fec_get_mac_addr(unsigned char *mac)
{
	struct spi_flash *env_flash;
	char *buffer, *cur_var, *val;
	int i, ret;
#if 0
	u32 val;
	int timeout = 1000;

	/* set this bit to open the OTP banks for reading */
	REG_WR(REGS_OCOTP_BASE, HW_OCOTP_CTRL_SET,
		BM_OCOTP_CTRL_RD_BANK_OPEN);

	/* wait until OTP contents are readable */
	while (BM_OCOTP_CTRL_BUSY & REG_RD(REGS_OCOTP_BASE, HW_OCOTP_CTRL)) {
		if (timeout-- < 0)
			return -ETIMEDOUT;
		udelay(100);
	}

	val = REG_RD(REGS_OCOTP_BASE, HW_OCOTP_CUSTn(0));
	mac[0] = (val >> 24) & 0xFF;
	mac[1] = (val >> 16) & 0xFF;
	mac[2] = (val >> 8) & 0xFF;
	mac[3] = (val >> 0) & 0xFF;
	val = REG_RD(REGS_OCOTP_BASE, HW_OCOTP_CUSTn(1));
	mac[4] = (val >> 24) & 0xFF;
	mac[5] = (val >> 16) & 0xFF;

#else
	env_flash = spi_flash_probe(0, 0, 1000000, SPI_MODE_3);
	if (!env_flash)
		goto err_probe;

	buffer = malloc (0x800);
	ret = spi_flash_read(env_flash, 0x7f800, 0x800, buffer);
	if (ret)
		goto err_read;

	/* sanity checks... */
	buffer[0x7fe] = buffer[0x7ff] = '\0';
	for (i = 0; i < 0x7ff; i++) {
		if (buffer[i] > 127) {
			buffer[i] = buffer[i+1] = '\0';
			break;
		}
	}
	cur_var = buffer;
	while ((val = strchr (cur_var, '=')) != NULL) {
		*val = '\0';
		setenv (cur_var, val + 1);
		cur_var = strchr (val + 1, '\0') + 1;
	}

err_read:
	spi_flash_free(env_flash);
	if (buffer)
		free (buffer);

err_probe:
	if (!eth_getenv_enetaddr("ethaddr", mac)) {
		/* a0:dc:04:00:00:00  */
		/* \______/ only for Becker use! */
		mac[0] = 0xa0;
		mac[1] = 0xdc;
		mac[2] = 0x04;
		mac[3] = 0x00;
		mac[4] = 0x00;
		mac[5] = 0x00;

		printf("Dummy MAC: %02x:%02x:%02x:%02x:%02x:%02x\n",
		       mac[0], mac[1], mac[2], mac[3], mac[4], mac[5]);
	}
#endif
	return 0;
}
#endif

void enet_board_init(void)
{
	/* Set up ENET pins */
	pin_set_group(&enet_pins);

	/* Power on the external phy */
	pin_gpio_set(PINID_PWM4, 1);
	pin_gpio_direction(PINID_PWM4, 1);
	pin_set_type(PINID_PWM4, PIN_GPIO);

	/* Reset the external phy */
	pin_gpio_set(PINID_ENET0_RX_CLK, 0);
	pin_gpio_direction(PINID_ENET0_RX_CLK, 1);
	pin_set_type(PINID_ENET0_RX_CLK, PIN_GPIO);
	udelay(200);
	pin_gpio_set(PINID_ENET0_RX_CLK, 1);
}

#ifdef CONFIG_MXS_NAND
#include <linux/mtd/nand.h>
extern int mxs_gpmi_nand_init(struct mtd_info *mtd, struct nand_chip *chip);

int board_nand_init(struct mtd_info *mtd, struct nand_chip *chip)
{
	pin_set_group(&gpmi_pins);
	return mxs_gpmi_nand_init(mtd, chip);
}
#endif

int checkboard(void)
{
	printf("Board: Becker CC51\n");

	return 0;
}


int spi_cs_is_valid(unsigned int bus, unsigned int cs)
{
	return bus == 0;
}

void spi_cs_activate(struct spi_slave *slave)
{
	pin_gpio_set (PINID_SSP2_SS0, 0);
}

void spi_cs_deactivate(struct spi_slave *slave)
{
	pin_gpio_set (PINID_SSP2_SS0, 1);
}

int  spi_bitbang_read ()       { return (pin_gpio_get (PINID_SSP2_MISO) != 0); }
void spi_bitbang_sda (int bit) { pin_gpio_set (PINID_SSP2_MOSI, bit); }
void spi_bitbang_scl (int bit) { pin_gpio_set (PINID_SSP2_SCK, bit); }

int  i2c_bitbang_read ()       { return (pin_gpio_get (PINID_I2C0_SDA) != 0); }
void i2c_bitbang_active ()     { pin_gpio_direction (PINID_I2C0_SDA, 1); }
void i2c_bitbang_tristate ()   { pin_gpio_direction (PINID_I2C0_SDA, 0); }
void i2c_bitbang_sda (int bit) { pin_gpio_set (PINID_I2C0_SDA, bit); }
void i2c_bitbang_scl (int bit) { pin_gpio_set (PINID_I2C0_SCL, bit); }


#define COLOR16(r,g,b,v) (((((r)*(v))>>11)<<11) |\
                          ((((g)*(v))>>10)<<5) |\
                          ((((b)*(v))>>11)<<0))

static void cc51_custom_logo_render (u8 r, u8 g, u8 b)
{
	const uint8_t *src = becker_logo.rle_data;
	u16 *pixels = (u16*) gd->fb_base;
	u16 *dest, *p0;
	uint8_t count, i;

	memset (pixels, '\x00', 480 * 272 * 2);

	dest = p0 = pixels + (272-20-becker_logo.height)/2*480 + (480-becker_logo.width)/2;

	count = 1;
	while (count) {
		count = *src;
		src++;
		for (i = 0; i < (count & 0x7f); i++) {
			*dest = COLOR16(r,g,b,*src);
			dest++;
			if (!(count & 0x80))
				src++;
			if ((dest - p0) % 480 == becker_logo.width)
				dest += (480 - becker_logo.width);
		}
		if (count & 0x80)
		  src++;
	}

#if 0
	/* backlight on */
	pin_gpio_set (PINID_PWM1, 0);
	pin_gpio_direction(PINID_PWM1, 1);
#endif
}

/* This is a trivial atoi implementation since we don't have one available */
int atoi(char *string)
{
	int length;
	int retval = 0;
	int i;
	int sign = 1;

	length = strlen(string);
	for (i = 0; i < length; i++) {
		if (0 == i && string[0] == '-') {
			sign = -1;
			continue;
		}
		if (string[i] > '9' || string[i] < '0') {
			break;
		}
		retval *= 10;
		retval += string[i] - '0';
	}
	retval *= sign;
	return retval;
}


void cc51_custom_logo (void)
{
	cc51_custom_logo_render (0, 78, 135);
}

void
cc51_custom_progress (int progress, int maxprogress)
{
	int val = 0xff * progress / maxprogress;

	/* poor gamma 2.2 approximation */
	val = val <  24 ? 1 :
	      val <  59 ? val/4 - 4 :
	      val < 100 ? (val+1)/2 - 19 :
	      val < 187 ? val - 68 :
	      val < 255 ? val*2 - 254 :
	      255;

	writel ((val & 0xffff) << 16, PWM_BASE + PWM_ACTIVE1);
	writel (0x000e00ff, PWM_BASE + PWM_PERIOD1);
}

int do_cc51_logo(cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	int r, g, b;

	if (argc < 4) {
		r = 0;
		g = 78;
		b = 135;
	} else {
		r = atoi (argv[1]);
		g = atoi (argv[2]);
		b = atoi (argv[3]);
	}

	cc51_custom_logo_render (r, g, b);
	return 0;
}

U_BOOT_CMD(cc51logo, 4, 0, do_cc51_logo,
	"renders the cc51 logo to the screen",
	"<red> <green> <blue>"
);

int do_dumpvars (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	int i;
	char *base = NULL;
	char *val;

	if (argc < 1) {
		printf("specify base addr\n");
		return -EINVAL;
	}

	base = (char *) simple_strtoul(argv[1], NULL, 16);

	for (i = 2; i < argc; i++) {
		val = getenv (argv[i]);
		if (!val)
			continue;
		base += sprintf (base, "%s=%s%c", argv[i], val, 0);
	}

	return 0;
}

U_BOOT_CMD(dumpvars, 100, 0, do_dumpvars,
	"dumps the listed variables into a key=value\\0 memory segment",
	"<address> <variable name>..."
);

int do_restorevars (cmd_tbl_t *cmdtp, int flag, int argc, char *argv[])
{
	int i;
	char *base = NULL;
	char *val, *cur_var;

	if (argc < 1) {
		printf("specify base addr\n");
		return -EINVAL;
	}

	base = (char *) simple_strtoul(argv[1], NULL, 16);
	cur_var = base;

	if (argc <= 2) {
		while ((val = strchr (cur_var, '=')) != NULL) {
			*val = '\0';
			setenv (cur_var, val + 1);
			*val = '=';
			cur_var = strchr (cur_var, '\0') + 1;
		}
	} else {
		for (i = 2; i < argc; i++) {
			cur_var = base;
			while ((val = strchr (cur_var, '=')) != NULL) {
				*val = '\0';
				if (!strcmp (argv[i], cur_var))
					setenv (cur_var, val + 1);
				*val = '=';
				cur_var = strchr (cur_var, '\0') + 1;
			}
		}
	}

	return 0;
}

U_BOOT_CMD(restorevars, 100, 0, do_restorevars,
	"restores the listed (or all) variables from a key=value\\0 memory segment",
	"<address> [<variable name>...]"
);
